#include "symtab.h"
#include "dtype.h"
#include "symbol.h"
#include "core.h"
#include "mipssymtab.h"
#include <sun/elf.h>
SRCFILE("mipssymtab.c")

int ccdemangle(char**,char* =0,int =0);

void FileDesc::init(MipsSymTab *symtab)
{
	issBase += (long)symtab->locstrings;
	cbLineOffset += (long)symtab->lines;
	rfdBase =  symtab->rfd + (long) rfdBase;
	isymBase = symtab->locsym + (long) isymBase;
	for( int j = 0; j < csym; ++j)
		isymBase[j].init((long)issBase);
	iauxBase = symtab->auxsym + (long) iauxBase;
}

// return Source node for this File with linked sub-structure of Func stubs
Source *FileDesc::src(MipsSymTab *symtab, Source *sc)
{
	if (rss == -1)
		return 0;
	char *path = issBase+rss;			// source file name
	static char *lastp;
	ProcDesc *pds = symtab->procdesc + ipdFirst;	// first proc
	Source *s = 0, *rets = 0;
	LocSym *lsym;
	Block *fake = 0;
	Var *fst = 0;
	if (!cpd)
		return s;
	if (glevel != 2) {				// Has -g
		for (; sc; sc = (Source*)sc->rsib)
			if (!strcmp(path, sc->_text))
				break;
		if (!sc) {
			rets = s = new Source(symtab, 0, path, 0);
			char *cp = path + strlen(path) + 1;
			if (!strcmp("</4Debug/>", cp)) {
				cp += strlen(cp) + 1;
				char *dir = strchr(cp, ':');
				if (dir) {
					dir++;
					cp = strrchr(dir, '/');
					if (cp) {
						cp[1] = 0;
						s->dir = dir;
					}
				}
			}
		} else {
			s = sc;
			if (fst = s->blk->var)
				while (fst->rsib)
					fst = (Var *)fst->rsib;
		}
		lsym = isymBase;
	}
	adr -= pds->adr;
	for( int i = 0; i < cpd; ++i ){
		ProcDesc *p = pds+i;
		++FunctionStubs;
		LocSym *fsym = &isymBase[p->isym];
		char *name = fsym->iss;
		int demangled = ccdemangle(&name);
		Func *func = new Func(symtab, name, s, p->lnLow);
		if (demangled) {
			name = fsym->iss;
			ccdemangle(&name, 0, 1);
			func->namewithargs = name;
		}
		func->begin = ipdFirst + i;
		func->range.lo = adr + p->adr;
		func->lines.hi = p->lnHigh;
		func->type = symtab->gatherdtype(isymBase+(p->isym), this);
		if (s) {
			while (lsym < fsym) {
				if (lsym->st() == stStatic)
					symtab->gathervar(lsym, &fst, s->blk,
						U_FST, this);
				lsym++;
			}
			lsym = isymBase + iauxBase[fsym->index()].bits;
		} else {
			if (!fake) fake = symtab->fakeblk();
			func->_blk = fake;
		}
	}
	return rets;
}

void ProcDesc::init(FileDesc *f)
{
	cbLineOffset += (long)f->cbLineOffset;
}

void LocSym::init(long base)		{ iss += base; }
void ExtSym::init(long base)		{ locsym.init(base); }

int AuxSym::pccbt()			// convert type to pcc
{
	switch( bt() ){
	case btChar: 	return CHAR;
	case btUChar: 	return UCHAR;
	case btShort: 	return SHORT;
	case btUShort: 	return USHORT;
	case btNil:
	case btAdr:
	case btInt:
	case btLong: 	return LONG;
	case btUInt:
	case btULong: 	return ULONG;
	case btFloat: 	return FLOAT;
	case btDouble: 	return DOUBLE;
	case btStruct: 	return STRTY;
	case btUnion: 	return UNIONTY;
	case btEnum: 	return ENUMTY;
	}
	return UNDEF;
}

// convert type qualifier to pcc
int AuxSym::pcctq(int i)
{
	switch( tq(i) ){
	case tqPtr:   return PTR;
	case tqArray: return ARY;
	case tqProc:  return FTN;
	}
	return 0;
}

// return filedesc for file in which given proc belongs
FileDesc *MipsSymTab::proc2filed(ProcDesc *p)
{
	FileDesc *f = filedesc;
	long procno = p - procdesc;

	for( long i = 0; i < shdr.ifdMax; ++i ){
		if( procno < f->cpd )
			return f;		// should return here
		procno -= f->cpd;
		++f;
	}
	return filedesc;			// should never get here
}

// return filedesc for file in which given local symbol belongs
FileDesc *MipsSymTab::locsym2filed(LocSym *s)
{
	for( long i = 0; i < shdr.ifdMax; ++i ){
		long delta = s - filedesc[i].isymBase;
		if( delta >= 0
		 && delta < filedesc[i].csym )
			return filedesc+i;	// should return here
	}
	return filedesc;			// should never get here
}

// build and return linked list of type members starting at locsym[u->begin]
Var *MipsSymTab::gatherutype(UType *u)	
{
	Var *first = 0, *v = 0;
	int nest = 0;
	int isnotenum = u->type.pcc != ENUMTY;
	for( LocSym *s = locsym+u->begin; ; ++s ){
		switch( s->st() ){
		case stTypedef: // Know type, but not members
			if (!nest)
				return first;
			break;
		case stEnd:
			if( --nest <= 0 ) {
				++UTypeGathered;
				uncfront(first, u->_text);
				return first;
			}
			break;
		case stStruct:
		case stUnion:
		case stEnum:
		case stBlock:
			++nest;
			break;
		case stMember:
			if( nest == 1 ){
				gathervar(s, &v, 0, U_MOT);
				if (isnotenum) {
					int type = v->type.pcc;
					if (type == BITS || type == UBITS) {
						v->range.lo =
						  (v->range.lo & 0xFFFFFFE0) +
						  32 - v->type.dim -
						  (v->range.lo & 0x1F);
					} else
						v->range.lo /=8 ;
				}
				if( !first ) first = v;
			}
			break;
		}
	}
}

UType *MipsSymTab::utypesym(LocSym *s, DType d)
{
	UType *u = (UType*)idtosym(U_UTYPE, s->iss, 0);
	if (u) {
		if (u->type.pcc != d.pcc)	// Patch type
			u->type.pcc = d.pcc;
	} else {	// No member info - make dummy entry
		u = new UType(this, s-locsym, 0, s->iss);
		u->range.lo = 0;
		u->type.pcc = d.pcc;
		u->type.univ = u;
		u->rsib = utype;
		utype = u;
	}
	return u;
}

int MipsSymTab::auxextended(AuxSym *a, FileDesc *f, LocSym **sr, FileDesc **fr)
{
	long rfd = a->rfd();
	long ix = a->index();
	if (rfd == 0xFFF)			// the illusive extended
		rfd = a[1].index();		// file index
	if (rfd >= 0 && rfd < 4095) {
		if (rfd == 0 && ix == 0)	// bogus
			return 0;
		*fr = filedesc + f->rfdBase[rfd];
	} else
		*fr = f;
	*sr = (*fr)->isymBase + ix;
	return 1;
}

// build C data type description
DType MipsSymTab::gatherdtype(LocSym *s, FileDesc *f)
{
	DType d, *dp;
	LocSym *stype;
	FileDesc *tyf;
	int pcc;
	d.pcc = LONG;				// default
	long ix = s->index();			// index to aux descriptor
	if( ix == 0xFFFFF )			// missing
		return d;
	// lazy caller does not know the FileDesc
	if( !f ) f = locsym2filed(s);
	AuxSym *aux = f->iauxBase + ix;
	if( s->st() == stProc )			// skip the endref
		++aux;
	AuxSym *a = aux + 1;			// steps thru following records
	long basetype = aux->bt();
	if (basetype == btTypedef) {
		if (!auxextended(a, f, &stype, &tyf))
			return d;
		d = gatherdtype(stype, tyf);
		a += 2;
	} else {
		if (basetype == 63)		// missing
			return d;
		else
			d.pcc = aux->pccbt();
		if (aux->isbitfield()) {
			d.dim = (int)a->bits;
			if (d.pcc == UCHAR || d.pcc == USHORT || d.pcc == ULONG)
				d.pcc = UBITS;
			else
				d.pcc = BITS;
		}
		if (d.pcc == STRTY || d.pcc == UNIONTY || d.pcc == ENUMTY) {
			if (!auxextended(a, f, &stype, &tyf))
				return d;
			if (stype->st() == stTypedef) {
				ix = stype->index();
				if (ix == 0xFFFFF ||
			   	    !auxextended(&tyf->iauxBase[ix+1],tyf,
					&stype,&tyf))
					return d;
			}
			d.univ = utypesym(stype, d);
			a += 2;
		}
	}
	// check each type qualifier
	for( int i = 0; i <= 3 && aux->tq(i); ++i ){
		if (pcc = aux->pcctq(i)) {
			dp = new DType;
			*dp = d;
			d.pcc = pcc;
			if( d.isary() ){		// find array dimension
				d.dim = a[3].dnHigh()+1; // max index -> size
				a += 5;
			}
			d.univ = dp;
		}
	}
	if( s->st() == stProc ){		// bt => ()bt
		dp = new DType;
		*dp = d;
		d.univ = dp;
		d.pcc = FTN;
	}
	return d;
}

MipsSymTab::MipsSymTab(Core* c, int fd, SymTab *i, long r):SymTab(c, fd, i, r)
{
}

MipsSymTab::~MipsSymTab()
{
	if( raw ) delete [] raw;
	if( iself && lines ) delete [] lines;
}

char *MipsSymTab::gethdr()
{
	char magic[4];
	char *virt, *err;

	if( lseek(fd, 0L, 0) == -1
	 || !ReadOK(fd, (char*)magic, sizeof(magic)) )
		return SysErr("file header: ");
	lseek(fd, 0L, 0);
	if (!strncmp(magic, "\177ELF", 4))
		err = getelfhdr(&virt);
	else
		err = getcoffhdr(&virt);
	if (err)
		return err;
	locstrings = virt + shdr.cbSsOffset;
	extstrings = virt + shdr.cbSsExtOffset;
	locsym = (LocSym*)(virt + shdr.cbSymOffset);
	auxsym = (AuxSym*)(virt + shdr.cbAuxOffset);
	filedesc = (FileDesc*)(virt + shdr.cbFdOffset);
	extsym = (ExtSym*)(virt + shdr.cbExtOffset);
	rfd = (long*)(virt + shdr.cbRfdOffset);
	long i;
	for( i = 0; i < shdr.iextMax; ++i )
		extsym[i].init((long)extstrings);
	for( i = 0; i < shdr.ifdMax; ++i )
		filedesc[i].init(this);
	procdesc = (ProcDesc*)(virt + shdr.cbPdOffset);
	for( i = 0; i < shdr.ifdMax; ++i ){
		FileDesc *f = filedesc+i;
		for( int j = f->ipdFirst; j < f->ipdFirst+f->cpd; ++j )
			procdesc[j].init(f);
	}
	char *high = (char*)&lines[shdr.cbLine];
	for( i = shdr.ipdMax-1; i >= 0; --i )
		if( procdesc[i].cbLineOffset ){
			procdesc[i].ilineMax = high - procdesc[i].cbLineOffset;
			high = procdesc[i].cbLineOffset;
		}
	entries = shdr.ifdMax;
	return 0;
}

char *MipsSymTab::getelfhdr(char **virt)
{
	char *err = 0;
	Elf32_Ehdr elfhdr;

	if (!ReadOK(fd, (char*)&elfhdr,sizeof elfhdr))
		return SysErr("ELF header: ");
	if (elfhdr.e_ident[EI_MAG0] != ELFMAG0 ||
	    strncmp((char*)&elfhdr.e_ident[EI_MAG1], "ELF", 3) ||
	    elfhdr.e_ident[EI_CLASS] != ELFCLASS32 ||
	    elfhdr.e_ident[EI_DATA] != ELFDATA2MSB ||
	    elfhdr.e_type != ET_EXEC ||
	    elfhdr.e_machine != EM_MIPS)
		return "symbol table: not MIPS ELF executable";
	int nsect = elfhdr.e_shnum;
	Elf32_Shdr *sections = new Elf32_Shdr[nsect];
	if (lseek(fd, elfhdr.e_shoff, 0) == -1 ||
	    !ReadOK(fd, (char*)sections, sizeof(*sections) * nsect)) {
		delete [] sections;
		return SysErr("ELF section headers: ");
	}
	int nbytes = (int)sections[elfhdr.e_shstrndx].sh_size;
	char *sstrings = new char[nbytes];
	if (lseek(fd, sections[elfhdr.e_shstrndx].sh_offset, 0) == -1 ||
	    !ReadOK(fd, sstrings, nbytes)) {
		delete [] sstrings;
		delete [] sections;
		return SysErr("ELF section strings: ");
	}
	Elf32_Shdr *s, *se = &sections[nsect];
	for (s = sections + 1; s < se; s++) {
		s->sh_name += (long)sstrings;
		if (!strcmp(s->sh_name, ".mdebug"))
			break;
	}
	if (s == se || !(nbytes = (int)s->sh_size)) {
		delete [] sstrings;
		delete [] sections;
		return "No symbol table";
	}
	raw = new char[nbytes];	
	if (lseek(fd, s->sh_offset, 0) == -1 ||
	    !ReadOK(fd, raw, nbytes))
		err =  SysErr("symbol table entries: ");
	else {
		shdr = *(SymHdr *)raw;
		*virt = raw - s->sh_offset;
		lines = new char[shdr.cbLine];
		if (lseek(fd, shdr.cbLineOffset, 0) == -1 ||
		    !ReadOK(fd, lines, (int)shdr.cbLine))
			err =  SysErr("symbol table line entries: ");
	}
	delete [] sstrings;
	delete [] sections;
	iself = 1;
	return err;
}

char *MipsSymTab::getcoffhdr(char **virt)
{
	FileHdr fhdr;
	OptHdr ohdr;

	if(!ReadOK(fd, (char*)&fhdr, sizeof fhdr) ||
	   !ReadOK(fd, (char*)&ohdr, sizeof ohdr) )
		return SysErr("file hdr/opt hdr: ");
	if( fhdr.magic != 0x0160 &&
	    fhdr.magic != 0x0163 &&
	    fhdr.magic != 0x0140 )
		return "bad magic";
	if( !fhdr.nscns )
		return "no sections";
	lseek(fd, fhdr.symptr, 0);
	if( !ReadOK(fd, (char*)&shdr, sizeof shdr) )
		return SysErr("symbolic header: ");
	long ts = shdr.tablesize();
	if( !ts )
		return "zero table size";
	lseek(fd, shdr.cbLineOffset, 0);
	raw = new char[ts];	
	if( !ReadOK(fd, raw, (int)ts) )
		return SysErr("line #s thru externals: ");
	lines = raw;
	*virt = raw - shdr.cbLineOffset;
	return 0;
}

// build func sub-tree starting
// at procdesc[func->begin]
Block *MipsSymTab::gatherfunc(Func *func)
{
	register Block *ablk, *lblk;		// args block, locals block
	Var *arg = 0, *lcl = 0;			// current arg, current local
	ProcDesc *p = procdesc + func->begin;
	FileDesc *f = proc2filed(p);
	LocSym *s = f->isymBase + p->isym;	// first local symbol in func
//	if( strcmp(func->_text, s->iss) )	// id should match
//		return 0;
	ablk = new Block(this,    0, 0, sf("%s.args",s->iss));
	lblk = new Block(this, ablk, 0, sf("%s.lcls",s->iss));
	ablk->child = lblk;			// locals fit below args
	++FunctionGathered;
	int nest = 0;
	for( ; ; ++s ){
		int isreg = s->isreg();
		switch( s->st() ){
		case stEnd:
			if (nest-- <= 0)
				goto BreakOut;
			break;
		case stParam:
			gathervar(s, &arg, ablk, isreg ? U_REG : U_ARG);
			if (f->lang == 2) { // Fortran - fix entry
				if (arg->type.isintegral()||arg->type.isreal()){
					DType *t = new DType(arg->type);
					arg->type = t->incref();
				} else if (arg->type.isary())
					arg->type.pcc = PTR;
			}
			break;
		case stLocal:
			gathervar(s, &lcl, lblk, isreg ? U_REG : U_AUT);
			break;
		case stStatic:
			gathervar(s, &lcl, lblk, U_STA);
			break;
		case stStruct:	// Struct declaration in function
		case stUnion:
		case stEnum:
		case stBlock:
			++nest;
			break;
		}
	}
BreakOut:			// build a list of Stmts by scanning the
	Stmt *stmt = 0;		// packed line number representation
	long line = p->lnLow;	// mips asm guide page 11-9
	long pc = func->range.lo;
	ablk->range.lo = pc;
	char *l = p->cbLineOffset;
	for( int j = 0; j < p->ilineMax; ++j ){
		long delta = (l[j]>>4)&0xF;
		if( delta&0x8 ) delta |= -16;
		long count = (l[j]&0xF) + 1;		//  1 .. 16
		if( delta == -8 ){
			delta = (l[j+1]<<8) | l[j+2];
			j += 2;
		}
		if (delta || !stmt || stmt == ablk->stmt) {
			line += delta;
			stmt = new Stmt(this, lblk, stmt);
			if( !ablk->stmt ) ablk->stmt = stmt;
			stmt->lineno = (short)line;
			stmt->range.lo = pc;
		}
		pc += count*4;
		stmt->range.hi = ablk->range.hi = pc;
	}
	uncfront( ablk->var, (char *)0 );
	uncfront( lblk->var, (char *)0 );
	return ablk;
}

void MipsSymTab::gathervar(LocSym *s, Var **v, Block *b, UDisc d, FileDesc *f)
{
	IF_LIVE( !v ) return;
	*v = new Var(this, b, *v, d, s->iss);
	if( b && !b->var ) b->var = *v;
	(*v)->range.lo = s->value;
	(*v)->type = gatherdtype(s, f);
}

// build globals block, SymTab::_blk
// make registers, then scan externals
void MipsSymTab::gatherglobals()
{
	Func *fn;
	Block *fake = fakeblk();
	Var *glb = globregs(_blk, _core->nregs() );

	for( int i = 0; i < shdr.iextMax; ++i ){
		LocSym *s = &(extsym[i].locsym);
		FileDesc *f = filedesc + extsym[i].ifd;
		if( s->iss )
			switch (s->st()) {
			case stGlobal:
				ccdemangle(&s->iss);
				gathervar(s, &glb, _blk, U_GLB, f);
				break;
			case stProc:
				ccdemangle(&s->iss);
				if( idtosym(U_FUNC, s->iss) )
					break;
				fn = new Func(this, s->iss, 0, 0);
				fn->range.lo = s->value;
				fn->_blk = fake;
				fn->type.pcc = LONG;
				break;
			}
	}
}

// Initial scan of symbol table.
Source *MipsSymTab::tree()
{
	Source	 *src = 0;
	int i;
	unsigned st;
	LocSym *e = &locsym[shdr.isymMax];

	for (LocSym *s = locsym; s < e; s++)
		if (s->sc() == scInfo &&
		    ((st = s->st()) == stStruct || st == stEnum ||
		      st == stUnion || st == stBlock) )
			gatherustub(s);
	for (i = 0; i < shdr.ifdMax; ++i) {
		Source *s = filedesc[i].src(this, src);
		if( s ){
			s->rsib = src;
			if( src ) src->lsib = s;
			src = s;
		}
	}
	gatherglobals();
	return src;
}

void MipsSymTab::gatherustub(LocSym *s)
{
	static int sindex = 1;
	ccdemangle(&s->iss);
	if (!s->iss || !*s->iss)
		s->iss = sf(".L%d", sindex++);
	UType *u = (UType*)idtosym(U_UTYPE, s->iss, 0);
	if (u)
		return;
	++UTypeStubs;
	u = new UType(this, s-locsym, 0, s->iss);
	u->range.lo = s->value;
	switch (s->st()) {
		case stStruct:
			u->type.pcc = STRTY;
			break;
		case stUnion:
			u->type.pcc = UNIONTY;
			break;
		case stEnum:
			u->type.pcc = ENUMTY;
			break;
		case stBlock:	// Non-SGI style
			u->type.pcc = STRTY;
			// Is it an enum ?
			if (s->value == 4 && (s+1)->st() == stMember) {
				FileDesc *f = locsym2filed(s+1);
				AuxSym *a = f->iauxBase + (s+1)->index();
				if (a->bt() == btNil)
					u->type.pcc = ENUMTY;
			}
			break;
	}
	u->type.univ = u;
	u->rsib = utype;
	utype = u;
}

// Change Func or Frame to grab this stuff and these functions can go away
long MipsSymTab::framesize(Func *func)
{
	return procdesc[func->begin].frameoffset;
}

long MipsSymTab::regmask(Func *func)
{
	return procdesc[func->begin].regmask;
}

long MipsSymTab::regoffset(Func *func)
{
	return procdesc[func->begin].regoffset;
}

short MipsSymTab::framereg(Func *func)
{
	return procdesc[func->begin].framereg;
}

short MipsSymTab::pcreg(Func *func)
{
	return procdesc[func->begin].pcreg;
}

int MipsSymTab::prologinstr(Func *func)
{
	ProcDesc *p = procdesc + func->begin;
	if (!p->ilineMax)
		return 0;
	return (p->cbLineOffset[0] & 0xF) + 1;
}
